function [data_rec, dsdr_rec, obj_func] = condat(data_clipped, param, data)
% CONDAT is an implementation of Generic proximal algorithm for Convex
% Optimization by Laurent Condat suited for the needs of signal declipping.


% setting different parameter for the simulation
paramsolver.maxit = 1000;     % maximum number of iterations
paramsolver.tau = 0.5;
paramsolver.sigma = 1/(3*paramsolver.tau); % only if F is a Parseval tight frame (FF* = 1)
paramsolver.ro = 0.99;        % extrapolation coeficient

paramsolver.verbose = 0;      % display parameter
paramsolver.comp_dsdr = 1;    % compute and store dSDR during iterations
paramsolver.comp_obj = 1;     % compute and store objective function values during iterations


% computing signal coefficients
c_clipped = frana(param.F, data_clipped);

% initialization of variables
u_r = frana(param.F, data_clipped .* param.masks.Mr);
u_h = data_clipped .* param.masks.Mh;
u_l = data_clipped .* param.masks.Ml;

% definition of soft thresholding
T = paramsolver.tau;
soft = @(z, T) sign(z).*max(abs(z)-T, 0);

% set starting point for convex optimization
c_old = c_clipped;

% precomputed coeffcicient of projection
coef = frana(param.F, (data_clipped .* param.masks.Mr));

% dsdr process
dsdr_rec = NaN(paramsolver.maxit,1);

% objective function process
obj_func = NaN(paramsolver.maxit, 1);

% iteration counter
cnt = 1;

while cnt <= paramsolver.maxit
    L_ur = u_r;
    L_uh = frana(param.F, u_h);
    L_ul = frana(param.F, u_l);
    
    suma = L_ur + L_uh + L_ul;
    c_tilde = soft((c_old - paramsolver.tau*suma), T);
    c_new = paramsolver.ro*c_tilde + (1-paramsolver.ro)*c_old;
    
    % reliable
    alfa = u_r + paramsolver.sigma*(2*c_tilde - c_old);
    
    syn = frsyn(param.F, alfa/paramsolver.sigma);
    syn = syn(1:param.Ls);
    syn(~param.masks.Mr) = 0;
    ana = frana(param.F, syn);
    proj_Kr = (alfa/paramsolver.sigma) - ana + coef;
    
    u_r_tilde = alfa - paramsolver.sigma*proj_Kr;
    u_r = paramsolver.ro*u_r_tilde + (1-paramsolver.ro)*u_r;
    
    % high
    extrapol = paramsolver.sigma*frsyn(param.F, (2*c_tilde - c_old));
    extrapol = extrapol(1:param.Ls);
    beta = u_h + extrapol;
    
    projKh = beta/paramsolver.sigma;
    Mh_below = param.masks.Mh & (projKh < param.theta);
    projKh(Mh_below) = param.theta;
    
    u_h_tilde = beta - paramsolver.sigma*projKh;
    u_h = paramsolver.ro*u_h_tilde + (1-paramsolver.ro)*u_h;
    
    % low
    beta = u_l + extrapol;
    
    projKl = beta/paramsolver.sigma;
    Ml_above = param.masks.Ml & (projKl > -param.theta);
    projKl(Ml_above) = -param.theta;
    
    u_l_tilde = beta - paramsolver.sigma*projKl;
    u_l = paramsolver.ro*u_l_tilde + (1-paramsolver.ro)*u_l;
    
    c_old = c_new;
    
    if paramsolver.comp_dsdr
        data_rec_tmp = postpad(frsyn(param.F, c_new), param.Ls); % reconstructed signal
        data_rec_tmp(param.masks.Mr) = data_clipped(param.masks.Mr); % replacing samples on reliable positions
        dsdr_rec(cnt) = sdr(data, data_rec_tmp) - sdr(data, data_clipped); % computing dSDR
    end
    
    if paramsolver.comp_obj
        obj_func(cnt) = norm(c_new, 1); % computing objective function (l1 norm of coefficients)
    end
    
    if paramsolver.verbose
        fprintf(' Iteration number: %u\n', cnt);
        if paramsolver.comp_dsdr
            fprintf(' SDR improvement: %e \n', dsdr_rec(cnt));
        end
        if paramsolver.comp_obj
            fprintf(' Objective function value: %e \n', obj_func(cnt));
        end
        fprintf('\n')
    end
    
    cnt = cnt+1;
end


data_rec = postpad(frsyn(param.F, c_new), param.Ls); % reconstructed signal

end

