function [coefs, L, extensions, period, difference] = segdtwt(signal,s,d,D_Lo,D_Hi)

% SEGDTWT function computes the segmented wavelet transform (multilevel)
% - version 1.4 (adapting more to the segmentwise real-time)
% We suppose that we are informed in the moment the last segment is read
%
% signal.......signal to be transformed segment-by-segment
% s............segment length
% d............wavelet decomposition depth
% D_Lo,D_Hi....wavelet decomposition filters
%
% coefs........wavelet coefficients - in one cell array, but computed
%              segment-by-segment. The first cell contains the finest
%              details, the last one contains the approximations
% L............matrix containing numbers of coefficients computed from the individual segments.
%              Size (d+2)x(number of segments). The first row is the number of samples per segment.
% extensions...matrix of left and right extensions throughout the computation
%              Each column is of form [n Lmax(n) Rmin(n)]', where n is the
%              segment number, n=1,...,number of segments-1. The last column
%              contains the total right extension (by zeros).
% period.......length of the basic period of the extension lengths
% difference...the difference between the length of the
%              last segment and the length of the respective right
%              extension; if it would be negative, it is set to zero
%             
% Example
%   db2_D_Lo = wfilters('db2','ld');  db2_D_Hi = wfilters('db2','hd');
%   [coefs, L] = segdtwt(1:80, 30, 2, db2_D_Lo, db2_D_Hi)

% (c) 2004-2008 Pavel Rajmic
% Dept. of Telecommunications, FEEC, Brno University of Technology, Czech Republic

% last revision 2008-Sep-11


%% initialization of counters etc.
last_segment = 0;           % boolean...last segment received?
seg_no = 1;                 % number of current segment
m = length(D_Lo);           % filter length (orthogonal)
extens_len = hmcs(d,m);     % extension length

coefs = {};                 % preparing a cell array structure in which the resulting coefficients will be stored
coefs(d+1) = {[]};          % number of cells will be d+1
L = zeros(d+2,1);           % preparing the first column
difference = 0;             % difference used in case that the right extension is bigger than the length of the last segment

rmin_vec = rmin(d,s);       % right extensions - precomputation of one period, to be read modulo further
lmax_vec = lmax(rmin_vec,extens_len);   % left extensions
period   = length(rmin_vec);% length of the period
extensions = zeros(3,1);    % auxiliary output

%% receive the first segment
[AKTUAL, last_segment] = read_segment(signal,s,seg_no);
L(1,1) = length(AKTUAL);
% Extend AKTUAL from the left side by r(d) zeros
%AKTUAL = [zeros(1,extens_len)  AKTUAL];
AKTUAL = extend(AKTUAL,'left','zeros',extens_len);
%vystupni matice
extensions(1,1) = 1;
extensions(2,1) = lmax_vec(1); % always equals to 'extens_len'


%%%%%%%%%%%%%%%%%%
%% loop over all the segments but the last
while last_segment == 0
    % Read NEXT
    [NEXT, last_segment] = read_segment(signal, s, seg_no+1);
    L(1,seg_no+1) = length(NEXT);
    
    % delka segmentu vcetne leveho prodlouzeni
    len_AKTUAL = length(AKTUAL);
%     % Spocitej Lmax pro NEXT
%     Lmax = max_left_ext( len_AKTUAL, d, m );
%     % Rmin pro AKTUAL = r(d)-Lmax
%     Rmin = extens_len - Lmax;
    % Spocitej Rmin pro AKTUAL (precomputed, modulo reading)
    Rmin = rmin_vec(mod(seg_no-1,period)+1);
    % Spocitej Lmax pro NEXT
    Lmax = lmax_vec(mod(seg_no,period)+1);
    % filling the auxiliary output matrix
    extensions(1,seg_no) = seg_no;
    extensions(2,seg_no+1) = Lmax; 
    extensions(3,seg_no) = Rmin;
    
    % Prodluz AKTUAL zprava o Rmin vzorku z NEXT
    % (je ale nutne osetrit pripad, kdy v poslednim segmentu je Rmin je
    % vetsi nez jeho delka)
    if last_segment
        difference = max([0 Rmin - L(1,seg_no+1)]);
%         difference = Rmin - L(1,seg_no+1);
        if difference > 0
            %AKTUAL = [AKTUAL  NEXT(1:L(1,seg_no+1)) zeros(1,difference)];
            AKTUAL = extend(AKTUAL,'right',NEXT(1:L(1,seg_no+1)));
            AKTUAL = extend(AKTUAL,'right','zeros',difference);
        else
            %AKTUAL = [AKTUAL  NEXT(1:Rmin)];
            AKTUAL = extend(AKTUAL,'right',NEXT(1:Rmin));
        end
    else
        %AKTUAL = [AKTUAL  NEXT(1:Rmin)];
        AKTUAL = extend(AKTUAL,'right',NEXT(1:Rmin));
    end
    % NEXT = Prodluz NEXT zleva o Lmax vzorku z AKTUAL
    %NEXT = [ AKTUAL(len_AKTUAL-Lmax+1:len_AKTUAL)  NEXT];
    NEXT = extend(NEXT,'left',AKTUAL(len_AKTUAL-Lmax+1:len_AKTUAL));
    
    %Spocitej d-urovnovou (modifikovanou) WT z AKTUAL
    coefs_AKTUAL = mlwd(AKTUAL,d,D_Lo,D_Hi);

    %Usekni prebyvajici casti u vypoctenych detailu (pouze zleva)
    coefs_AKTUAL = cut_redundant(coefs_AKTUAL,m,'left');

    %Vystup prislusny segmentu cislo "seg_no"
    for ind = 1:d+1
        % ukladani poctu koeficientu do matice
        L(ind+1, seg_no) = length(coefs_AKTUAL{ind});
%         % overovani vzorce
%         pomoc(ind+1, seg_no) = no_of_coefficients_SegWT(length(AKTUAL),ind,m) - hmcs(d-ind,m);
        % skladam za sebe stavajici koeficienty a prave vypoctene
        coefs{ind} = [coefs{ind} coefs_AKTUAL{ind}];
    end
    
    %Posun o jeden segment dale
    AKTUAL = NEXT;
    seg_no = seg_no+1;
end


%% processing of the last segment
% Prodluz AKTUAL (tj. posledni) zprava na celkovou delku "leve_prodlouzeni+s"
len_AKTUAL = length(AKTUAL);    % delka segmentu vcetne jeho leveho prodlouzeni
if seg_no == 1                  % tj. posledni je zaroven prvnim segmentem
    no_zeros = s-(len_AKTUAL-extens_len);
else                            % neni prvni
    no_zeros = s-(len_AKTUAL-Lmax);
end
%[AKTUAL zeros(1,no_zeros)];
AKTUAL = extend(AKTUAL,'right','zeros',no_zeros);

%AKTUAL = Prodluz AKTUAL o r(d) nul zprava
%AKTUAL = [AKTUAL  zeros(1,extens_len)];
AKTUAL = extend(AKTUAL,'right','zeros',extens_len);

% filling the matrix of extensions
extensions(1,seg_no) = seg_no;
extensions(3,seg_no) = no_zeros + extens_len;
clear no_zeros

%Spocitej d-urovnovou WT z AKTUAL (stejne jako nize)
coefs_AKTUAL = mlwd(AKTUAL,d,D_Lo,D_Hi);

%Usekni prebyvajici casti u vypoctenych detailu (pouze zleva)
coefs_AKTUAL = cut_redundant(coefs_AKTUAL,m,'left');

% useknuti zprava - prvni
for ind = 1:d   % for detail coefficients
    % kolik by jich bylo byvalo spravne vypocteno v urovni "ind", kdybych nemusel pridavat nuly kvuli kratke delce segmentu
    hm_to_leave = floor((2^-ind)*(len_AKTUAL-0)); %radek XX
    %stary vypocet (vysledek ale stejny): hm_to_leave = no_of_coefficients(len_AKTUAL+extens_len, ind, m) - hmcs(d-ind,m)
    % leaving just the left part - for detail coefficients
    coefs_AKTUAL{ind} = coefs_AKTUAL{ind}( 1 : hm_to_leave );
end
coefs_AKTUAL{d+1} = coefs_AKTUAL{d+1}( 1 : hm_to_leave ); % for approximation coefficients

% useknuti zprava - druhe (zaridime stejne jako zleva, ale zde musime zprava)
coefs_AKTUAL = cut_redundant(coefs_AKTUAL,m,'right');
% for ind = 1:(d-1)
%     %How Many coefficients To be Cut in (ind)-th level from right
%     hm_to_cut = hmcs(d-ind,m);
%     %cutting the right part, leaving the rest
%     coefs_AKTUAL{ind} = coefs_AKTUAL{ind}( 1 : length(coefs_AKTUAL{ind})-hm_to_cut );
% end
%(tento cely blok useknuti se da samozrejme sloucit s predchozim (staci upravit radek XX)), pro kompatlibilitu s orezavanim koeficientu v drivejsih segmentech nechavam takto

% Hod vypoctene na vystup
for ind = 1:d+1
    % ukladani poctu koeficientu do matice
    L(ind+1, seg_no) = length(coefs_AKTUAL{ind});
%     pomoc(ind+1, seg_no) = no_of_coefficients_SegWT(len_AKTUAL+extens_len, ind, m);
    % skladam za sebe stavajici koeficienty a prave vypoctene
    coefs{ind} = [coefs{ind} coefs_AKTUAL{ind}];
end