function [signal_out, coefs, L, extensions, fragments, no_of_waiting] = rt_dtwt(signal_in,s,d,D_Lo,D_Hi,R_Lo,R_Hi)

% Real-time SegWT function computes the segmented wavelet transform (multilevel)
% (the forward and inverse algorithm are joined to make the real-time processing possible)
%
% The input-output delay is one segment (i.e. s samples) plus r(d) samples.
% This delay is filled in using zeros at the output.
%
% INPUT:
% signal_in.....signal to be transformed (and processed) segment-by-segment
% s.............segment length
% d.............wavelet decomposition depth
% D_Lo,D_Hi.....wavelet decomposition filters
% R_Lo,R_Hi.....wavelet reconstruction filters
%
% OUTPUT:
% signal_out....signal in time domain reconstructed segment-by-segment;
%               it has s+r(d) zeros at the beginning.
% AUXILIARY OUTPUT:
% coefs.........wavelet coefficients - in one cell array, but computed
%               segment-by-segment. The first cell contains the finest
%               details, the last one contains the approximations.
% L.............matrix containing numbers of coefficients computed from the individual segments.
%               Size (d+2)x(number of segments). The first row is the number of samples per segment.
% extensions....matrix of left and right extensions throughout the
%               computation of the forward part.
%               Each column is of form [n Lmax(n) Rmin(n)]', where n is the
%               segment number, n=1,...,(number of segments-1). The last column
%               contains the total right extension (by zeros).
% fragments.....vector of lengths of fragments within the reconstruction procedure
%               ([central overlap central ... central overlap])
%               The last number could be negative - that means that the last
%               piece had to be shortened by the respective number of
%               samples.
% no_of_waiting.vector containing the numbers od coeffeicients which have to wait
%               from one segment to another within the reconstruction because
%               of the correct number of samples at the output.
%               The i-th element represents the wainting samples between
%               segments (i-1) and (i). The first number is always 0.
%
% Example
%   db2_D_Lo = wfilters('db2','ld');  db2_D_Hi = wfilters('db2','hd');
%   db2_R_Lo = wfilters('db2','lr');  db2_R_Hi = wfilters('db2','hr');
%   [sigout, coefs, L, ext, frg, waiting] = rt_dtwt(1:475, 53, 3, db2_D_Lo, db2_D_Hi, db2_R_Lo, db2_R_Hi);

% (c) 2008-09 Pavel Rajmic
% Dept. of Telecommunications, FEEC, Brno University of Technology, Czech Republic

% last revision 2009-Jan-12


%% initialization of counters etc.
last = 0;                   % flag indicating if the last segment was received
seg_no = 1;                 % number of current segment
finish = 0;                 % flag indicating if the main cycle has to be run
m = length(D_Lo);           % filter length (orthogonal)
extens_len = hmcs(d,m);     % extension length

coefs = {};                 % preparing a cell array structure in which the resulting coefficients will be stored
coefs(d+1) = {[]};          % number of cells will be d+1
L = zeros(d+2,1);           % preparing the first column
difference = 0;             % difference used in case the right extension is bigger than the length of the last segment

rmin_vec = rmin(d,s);       % right extensions - precomputation of one period, to be read modulo further
lmax_vec = lmax(rmin_vec,extens_len);   % left extensions
period   = length(rmin_vec);% length of the period
extensions = zeros(3,1);    % auxiliary output

seg_signal = {};            % bunka pro docasne useky vysledneho signalu
waiting_samples = [];       % vzorky rekostruovaneho signalu, ktere budou cekat jedno kolo navic
overlap_len = extens_len;   % pocet prvku prekryti
signal_out = [];            % vystupni signal
fragments = [];             % pomocny vystup
no_of_waiting = 0;          % pocet koeficientu, ktere se schovavaji na prechodu segmentu kvuli synchronizaci
par = logical(0);           % parity (boolean)

%% the first output segment will always be zeros
signal_out = zeros(1,s);

%% receive the first segment
[AKTUAL, last] = read_segment(signal_in,s,seg_no);
L(1,1) = length(AKTUAL);
% AKTUAL = extend AKTUAL from the left side by r(d) zeros
AKTUAL = extend(AKTUAL,'left','zeros',extens_len);

%vystupni matice
extensions(1,1) = 1;
extensions(2,1) = lmax_vec(1); % always equals to 'extens_len'

%%%%%%%%%%%%%%%%%%
%% loop over all the segments but the last
while finish==0
    if last==0  % nebyl jeste nacten posledni segment
        % Read NEXT
        [NEXT, last] = read_segment(signal_in, s, seg_no+1);
        L(1,seg_no+1) = length(NEXT);

        % delka segmentu vcetne leveho prodlouzeni
        len_AKTUAL = length(AKTUAL);

        % Spocitej Rmin pro AKTUAL (precomputed, modulo reading)
        Rmin = rmin_vec(mod(seg_no-1,period)+1);
        % Spocitej Lmax pro NEXT
        Lmax = lmax_vec(mod(seg_no,period)+1);
        % filling the auxiliary output matrix
        extensions(1,seg_no) = seg_no;
        extensions(2,seg_no+1) = Lmax;
        extensions(3,seg_no) = Rmin;

        % AKTUAL = Prodluz AKTUAL zprava o Rmin vzorku z NEXT
        % (je ale nutne osetrit pripad, kdy v poslednim segmentu je Rmin je
        % vetsi nez jeho delka)
        if last
            difference = max([0 Rmin - L(1,seg_no+1)]);
            if difference > 0
                AKTUAL = extend(AKTUAL,'right',NEXT(1:L(1,seg_no+1)));
                AKTUAL = extend(AKTUAL,'right','zeros',difference);
            else
                AKTUAL = extend(AKTUAL,'right',NEXT(1:Rmin));
            end
        else
            AKTUAL = extend(AKTUAL,'right',NEXT(1:Rmin));
        end
        % NEXT = Prodluz NEXT zleva o Lmax vzorku z AKTUAL
        NEXT = extend(NEXT,'left',AKTUAL(len_AKTUAL-Lmax+1:len_AKTUAL));
        
        %Spocitej d-urovnovou (modifikovanou) WT z AKTUAL
        coefs_AKTUAL = mlwd(AKTUAL,d,D_Lo,D_Hi);

        %Usekni prebyvajici casti u vypoctenych detailu (zleva)
        coefs_AKTUAL = cut_redundant(coefs_AKTUAL,m,'left');

    else                % byl uz nacten posledni segment
        %%% processing of the last segment
        % Prodluz AKTUAL (tj. posledni) na celkovou delku "leve_prodlouzeni+s"
        len_AKTUAL = length(AKTUAL);    % delka segmentu vcetne jeho leveho prodlouzeni
        if seg_no == 1                  % tj. zaroven je prvnim segmentem
            no_zeros = s-(len_AKTUAL-extens_len);
        else                            % neni prvni
            no_zeros = s-(len_AKTUAL-Lmax);
        end
        AKTUAL = extend(AKTUAL,'right','zeros',no_zeros);

        %AKTUAL = Prodluz AKTUAL o r(d) nul zprava
        AKTUAL = extend(AKTUAL,'right','zeros',extens_len);

        % filling the matrix of extensions
        extensions(1,seg_no) = seg_no;
        extensions(3,seg_no) = no_zeros + extens_len;
        clear no_zeros

        %Spocitej d-urovnovou WT z AKTUAL (stejne jako nize)
        coefs_AKTUAL = mlwd(AKTUAL,d,D_Lo,D_Hi);

        %Usekni prebyvajici casti u vypoctenych detailu (pouze zleva)
        coefs_AKTUAL = cut_redundant(coefs_AKTUAL,m,'left');
        
        % useknuti zprava - prvni
        for ind = 1:d   % for detail coefficients
            % kolik by jich bylo byvalo spravne vypocteno v urovni "ind", kdybych nemusel pridavat nuly kvuli kratke delce segmentu
            hm_to_leave = floor((2^-ind)*(len_AKTUAL-0)); %radek XX
            % leaving just the left part - for detail coefficients
            coefs_AKTUAL{ind} = coefs_AKTUAL{ind}( 1 : hm_to_leave );
        end
        coefs_AKTUAL{d+1} = coefs_AKTUAL{d+1}( 1 : hm_to_leave ); % for approximation coefficients

        % useknuti zprava - druhe (zaridime stejne jako pri urezavani zleva, ale zde zprava)
        coefs_AKTUAL = cut_redundant(coefs_AKTUAL,m,'right');
        
        % indikator konce cyklu nastavit
        finish = 1;
    end

    % Koeficienty prislusne segmentu cislo "seg_no"
    for ind = 1:d+1
        % ukladani poctu koeficientu do matice
        L(ind+1, seg_no) = length(coefs_AKTUAL{ind});
        %%%%%%%%%%%%%%%%%%%%%%% PROCESSING HERE %%%%%%%%%%%%%%%%%%%%%%%%%%%
        % ....
        % skladam za sebe stavajici koeficienty a prave vypoctene
        coefs{ind} = [coefs{ind} coefs_AKTUAL{ind}];
    end
    
    %% The INVERSE PART begins here
    %zamena nazvu
    seg_coefs = coefs_AKTUAL;
    % rozsireni detailnich koeficientu nulami zleva
    for ind = 1:(d-1)
        seg_coefs{ind} = extend(seg_coefs{ind},'left','zeros',hmcs(d-ind,m));
    end

    % inverzni transformace jednoho segmentu
    seg_signal{seg_no} = mlwr(seg_coefs,d,R_Lo,R_Hi);
    seg_signal_len = length(seg_signal{seg_no});

    if seg_no == 1 % pokud je segment prvni
        current_output = zeros(1,extens_len);   % r(d) nul - zamerne zpozdeni vystupu
    else
        % vyvolani cekajicich vzorku a pricteni presahu z minuleho kola
        current_output = [waiting_samples seg_signal{seg_no}(1:overlap_len) + to_add];
    end

    % parity of the total length of read signal (even...0, odd...1)
    % - knowledge of this is sufficient for the correct cutting of the last
    % piece of the signal at the very end
    par = parity(par,L(1,seg_no));

    if last & finish==1
        % v pripade posledniho segmentu pripojime veskery zbytek k vystupu
        % (muze vyjit i prazdne), ale zaroven muze byt odebrano poslednich
        % nekolik vzorku signalu, aby byl vystupni signal stejne delky jako
        % vstupni
               
        index = mod(seg_no+3*period-2,period) + 1;
        difference = rmin_vec(index) - L(1,end);
        % correction (in fact, by this we avoid an IF branch)
        difference = max([par difference]); %
        
        indexes = (overlap_len+1):(seg_signal_len-m+2);
        current_output = [current_output seg_signal{seg_no}(indexes)];
        current_output = current_output(1:end-difference);
        
        % for the auxiliary output
        % the current fragment cannot be shorter that 0, however, negative
        % number at the last position means that the end of the signal has
        % been shortened
        fragments = [fragments (length(indexes)-difference)];
        
    else
        % Prostredni cast, ktera se nebude s nicim prekryvat
        % (teoreticky asi muze vyjit i prazdna).
        % U "prvniho" segmentu tato prostredni cast odpovida presne zacatku
        % rekonstruovaneho signalu.
        centerpart = seg_signal{seg_no}((overlap_len+1):(seg_signal_len-overlap_len));
        
        % pocet koeficientu, ktere museji cekat kvuli spravnemu poctu vystupnich
        % vzorku do dalsiho cyklu (vzdy nezaporne cislo); i-ty prvek
        % odpovida poctu cekajicich vzorku mezi segmenty (i-1) a (i)
        no_of_waiting(seg_no+1) = no_of_waiting(seg_no) + length(centerpart) + overlap_len - L(1,seg_no);
        % Pipojme prostedn st - ale obecn ne celou. Nkter
        % vzorky pokaj do ptho kola.
        current_output = [current_output centerpart(1:end-no_of_waiting(seg_no+1))];
        % ekajc
        waiting_samples = centerpart((end-no_of_waiting(seg_no+1)+1):end);
        % for the auxiliary output
        fragments = [fragments length(centerpart)];

        % presah - nove hodnoty k pricteni v pristim kole
        to_add = seg_signal{seg_no}((seg_signal_len-overlap_len+1):seg_signal_len); %r(d) hodnot
        fragments = [fragments length(to_add)];
        
    end

    % vystupni vektor, jehoz hodnoty se jiz nebudou menit
    signal_out = [signal_out current_output];
        
    % zvyseni citace segmentu
    seg_no = seg_no+1;
    %Posun o jeden segment dale
    AKTUAL = NEXT;

end

%% prozatimni urezani prvnich vystupnich nul kvuli kontrole vysledku
% length(signal_in)+s+extens_len
signal_out = signal_out(s+extens_len+1:end); %urezani prvnich nul z "vystupu"

chyba = norm(signal_in-signal_out)
% sum(fragments)